/******************************************************************************\
*       This is a part of the Microsoft Source Code Samples. 
*       Copyright (C) 1993 Microsoft Corporation.
*       All rights reserved. 
*       This source code is only intended as a supplement to 
*       Microsoft Development Tools and/or WinHelp documentation.
*       See these sources for detailed information regarding the 
*       Microsoft samples programs.
\******************************************************************************/

/*****************************************************************************\
*
* Module: dialogs.cpp
*
*   Contains dialog procs for the Windows debugging OAC Spy SDK applet.
*
* Functions:
*
*   MyDialogBox()
*   AboutDlgProc()
*   SelectWindowDlgProc()
*   SelectWindowCommand()
*   FillListBox()
*   AddOneWindow()
*   MakeWindowName()
*   FindHwndInListBox()
*   HighlightWindow()
*   SelectWindowUpdateInfo()
*   SelectWindowEnableFields()
*
* Comments:
*
\*****************************************************************************/

#include "oacspy.h"
#include <ole2.h>
#include <oleacc.h>

#include <commdlg.h>


#define DINV                3


PRIVATE HWND ghwndSpyingOnTemp;     // Temp when selecting hwnd to spy on.
PRIVATE HWND ghwndDlgBeingFilled = NULL;
PRIVATE BOOL bBorderOn = FALSE;
PRIVATE INT gcItemsSave;


PRIVATE BOOL SelectWindowCommand(HWND hwnd, INT nCmd, INT nNotifyCode);
PRIVATE VOID FillListBox(HWND hDlg, HWND hwndList, HWND hwnd);
void FillOACList(HWND, OACOBJINFO*);
BOOL CALLBACK AddOneWindow(HWND hwnd, HWND hwndList);
PRIVATE VOID MakeWindowName(HWND hwnd, LPSTR lpString, INT nStringLen);
PRIVATE INT FindHwndInListBox(HWND hwndList, HWND hSpyWnd);
PRIVATE VOID HighlightWindow(HWND hwnd, BOOL fDraw);
PRIVATE VOID SelectWindowUpdateInfo(HWND hDlg, HWND hwnd);
PRIVATE VOID SelectWindowEnableFields(HWND hwnd, BOOL fEnable);


CHAR szConsoleWindowClass[] = "ConsoleWindowClass";

/*****************************************************************************\
* MyDialogBox
*
* Puts up the specified dialog.
*
* Arguments:
*   INT idDlg          - The resource id of the dialog to display.
*   DLGPROC pfnDlgProc - The dialog proc to use.
*
* Returns:
*   The return value from DialogBox (whatever the dialog proc passes
*   to EndDialog).
\*****************************************************************************/

BOOL
MyDialogBox(
    INT idDlg,
    DLGPROC pfnDlgProc
    )
{
    return DialogBox(ghInst, MAKEINTRESOURCE(idDlg), ghwndSpyApp, pfnDlgProc);
}



/*****************************************************************************\
* AboutDlgProc
*
*   Dialog proc for the About box.
*
\*****************************************************************************/

BOOL CALLBACK
AboutDlgProc(
    HWND hwnd,
    UINT msg,
    WPARAM wParam,
    LPARAM lParam
    )
{
    switch (msg) {
	case WM_INITDIALOG:
	    return TRUE;

	case WM_COMMAND:
	    EndDialog(hwnd, IDOK);
	    break;
    }

    return FALSE;
}

BOOL CALLBACK
ObjectSelDlgProc(
    HWND hwnd,
    UINT msg,
    WPARAM wParam,
    LPARAM lParam
    )
{
    switch (msg) {
	case WM_INITDIALOG:
			ghdlgObjectSel = hwnd;
			ComboBox_AddString(GetDlgItem(hwnd, DID_OBJECTSELTYPE), "Keyboard focus");
			ComboBox_AddString(GetDlgItem(hwnd, DID_OBJECTSELTYPE), "Selection");
                        ComboBox_AddString(GetDlgItem(hwnd, DID_OBJECTSELTYPE), "Caret");
                        ComboBox_AddString(GetDlgItem(hwnd, DID_OBJECTSELTYPE), "Sementic change");
                        ComboBox_AddString(GetDlgItem(hwnd, DID_OBJECTSELTYPE), "Alert");
                        return TRUE;

	case WM_COMMAND:
			switch LOWORD((wParam))
			{
				case DID_OBJECTSELTYPE:
					if (HIWORD(wParam) == CBN_SELCHANGE)
					{
						// User wants to look at another type of OBJECTSEL
						int index = ComboBox_GetCurSel(GetDlgItem(hwnd, DID_OBJECTSELTYPE));
						FillOACList(GetDlgItem(hwnd, DID_OI),
							&grgoiObjectSel[index]);
					}
					break;

				case IDCANCEL:
			EndDialog(hwnd, IDOK); ghdlgObjectSel=0;
			break;
			}
    }

    return FALSE;
}



/*****************************************************************************\
* SelectWindowDlgProc
*
* Dialog proc for the Select Window dialog.  This dialog allows the user
* to select which window they want to spy on.
*
* Arguments:
*   HWND hwnd       - Window handle of the dialog.
*   UINT msg        - Message sent to window.
*   WPARAM wParam   - Message parameter.
*   LPARAM lParam   - Message parameter.
*
* Returns:
*   The value that the dialog proc should return, based on the processing
*   of the specific WM_COMMAND message received.
\*****************************************************************************/

BOOL CALLBACK
SelectWindowDlgProc(
    HWND hwnd,
    UINT msg,
    WPARAM wParam,
    LPARAM lParam
    )
{
    HWND hwndList;
    INT nIndex;

    UNREFERENCED_PARAMETER(lParam);

    switch (msg)
    {
	case WM_INITDIALOG:
		ghdlgSelectWindow = hwnd;
	    hwndList = GetDlgItem(hwnd, DID_SELWINLIST);
	    ghwndSpyingOnTemp = ghwndSpyingOn == HWND_ALL ? NULL : (ghwndSpyingOn == NULL ? NULL
		: GetParent(ghwndSpyingOn));
	    FillListBox(hwnd, hwndList, ghwndSpyingOnTemp);
	    nIndex = FindHwndInListBox(hwndList, ghwndSpyingOn == HWND_ALL ? NULL
		: ghwndSpyingOn);
	    SendMessage(hwndList, LB_SETCURSEL, nIndex, 0);
	    ghwndSpyingOnTemp = (HWND)SendMessage(hwndList, LB_GETITEMDATA, nIndex, 0);
	    SelectWindowUpdateInfo(hwnd, ghwndSpyingOnTemp);
		/*
	    CheckDlgButton(hwnd, DID_SELWINALLWINDOWS, gfSpyAll);
	    SelectWindowEnableFields(hwnd, !gfSpyAll);
		*/
	    // Select-all-window grays out the list box.
		// I am now unsetting it and hiding it all together.
	    CheckDlgButton(hwnd, DID_SELWINALLWINDOWS, FALSE);
	    SelectWindowEnableFields(hwnd, TRUE);
	    EnableWindow(GetDlgItem(hwnd, DID_SELWINALLWINDOWS), FALSE);
	    ShowWindow(GetDlgItem(hwnd, DID_SELWINALLWINDOWS), SW_HIDE);

	    SetFocus(hwnd);
	    HighlightWindow(ghwndSpyingOnTemp, TRUE);

	    return TRUE;

	case WM_NCLBUTTONDOWN:
	    if (wParam == HTCAPTION)
	    {
		//
		// The mouse is down for a move of the dialog, so clean up the
		// border stuff.
		//
		if (bBorderOn)
		    HighlightWindow(ghwndSpyingOnTemp, FALSE);
	    }

	    return FALSE;

	case WM_KEYDOWN:
	case WM_LBUTTONUP:
	case WM_NCLBUTTONUP:
	    //
	    // The mouse is up from a move of the dialog, so put up the
	    // border stuff again.
	    //
	    if (!bBorderOn)
		HighlightWindow(ghwndSpyingOnTemp, TRUE);

	    return FALSE;

	case WM_CANCELMODE:
	    return FALSE;

	case WM_COMMAND:
	    return SelectWindowCommand(hwnd, LOWORD(wParam), HIWORD(wParam));
    }

    return FALSE;
}



/*****************************************************************************\
* SelectWindowCommand
*
* Handles thw WM_COMMAND messages for the Select Window dialog.
*
* Arguments:
*   HWND hwnd       - Window handle of the dialog.
*   INT nCmd        - Command value.
*   INT nNotifyCode - The notify code.
*
* Returns:
*   The value that the dialog proc should return, based on the processing
*   of the specific WM_COMMAND message received.
\*****************************************************************************/

PRIVATE BOOL
SelectWindowCommand(
    HWND hwnd,
    INT nCmd,
    INT nNotifyCode
    )
{
    INT nIndex;
    HWND hwndList;
    CHAR rgString[32];

    switch (nCmd)
    {
	case IDOK:
/* Needed only for in-process spying.
	    SetWindowToSpyOn(IsDlgButtonChecked(hwnd, DID_SELWINALLWINDOWS) ?
		HWND_ALL : ghwndSpyingOnTemp);
*/

	    if (bBorderOn)
		HighlightWindow(ghwndSpyingOnTemp, FALSE);

	    EndDialog(hwnd, IDOK);
	    return TRUE;

	case IDCANCEL:
	    if (bBorderOn)
		HighlightWindow(ghwndSpyingOnTemp, FALSE);

	    EndDialog(hwnd, IDCANCEL);
	    return TRUE;

	case DID_SELWINLIST:
	    //
	    // User single clicked or doubled clicked in listbox -
	    //   Single click means select a window to spy on
	    //   Double click means enumerate all the children of that window.
	    //
	    hwndList = GetDlgItem(hwnd, DID_SELWINLIST);
	    switch (nNotifyCode)
	    {
		case LBN_SELCHANGE:
		    //
		    // Single click case. Select a window to spy upon.
		    //
		    // Get the window handle, set it as the window to spy on.
		    //

		    if (bBorderOn)
			HighlightWindow(ghwndSpyingOnTemp, FALSE);

		    nIndex = (INT)SendMessage(hwndList, LB_GETCURSEL, 0, 0);
		    ghwndSpyingOnTemp = (HWND)SendMessage(hwndList, LB_GETITEMDATA,
			nIndex, 0);
		    SelectWindowUpdateInfo(hwnd, ghwndSpyingOnTemp);

		    HighlightWindow(ghwndSpyingOnTemp, TRUE);

		    break;

		case LBN_DBLCLK:
		    //
		    // Double click case - first click has already been
		    // processed as single click. In this case, the user has
		    // requested to look at all the children of a given
		    // selection.
		    //
		    // Get the current selection, and check to see if it is the
		    // " [ parent.. ]" entry. If so, go up one level first.
		    //

		    SetCursor(LoadCursor(NULL, IDC_WAIT));

		    if (bBorderOn)
			HighlightWindow(ghwndSpyingOnTemp, FALSE);

		    nIndex = (INT)SendMessage(hwndList, LB_GETCURSEL, 0, 0);
		    ghwndSpyingOnTemp = (HWND)SendMessage(hwndList, LB_GETITEMDATA,
			nIndex, 0);
		    SendMessage(hwndList, LB_GETTEXT, nIndex, (LPARAM)rgString);

		    if (rgString[0] == ' ')
		    {
			// At top?  If so, we are done.
			if (ghwndSpyingOnTemp == NULL)
			{
			    SetCursor(LoadCursor(NULL, IDC_ARROW));
			    break;
			}

			ghwndSpyingOnTemp = GetParent(ghwndSpyingOnTemp);
		    }

		    SendMessage(hwndList, LB_RESETCONTENT, 0, 0);
		    FillListBox(hwnd, hwndList, ghwndSpyingOnTemp);

		    nIndex = FindHwndInListBox(hwndList, ghwndSpyingOnTemp);
		    SendMessage(hwndList, LB_SETCURSEL, nIndex, 0);
		    ghwndSpyingOnTemp = (HWND)SendMessage(hwndList, LB_GETITEMDATA,
			nIndex, 0);
		    HighlightWindow(ghwndSpyingOnTemp,TRUE);
		    SelectWindowUpdateInfo(hwnd, ghwndSpyingOnTemp);
		    SetCursor(LoadCursor(NULL, IDC_ARROW));
		    break;
	    }

	    break;

	case DID_SELWINALLWINDOWS:
	    SelectWindowEnableFields(hwnd,
		!IsDlgButtonChecked(hwnd, DID_SELWINALLWINDOWS));

	    break;
    }

    return FALSE;
}

void FillOACList(HWND hwndList, OACOBJINFO *poi)
{
	char buf[128];

	ListBox_ResetContent(hwndList);

        if (poi->hr != S_OK)
        {
            char szHR[64];
            HResultText(szHR, poi->hr);
            wsprintf(buf, "HRESULT error = %s", szHR);
            ListBox_AddString(hwndList, buf);
        }

        wsprintf(buf, "Rectangle = %ld, %ld, %ld, %ld",
		poi->xLeft, poi->yTop, poi->xWidth, poi->yHight);
	ListBox_AddString(hwndList, buf);

        wsprintf(buf, "Name = %s", poi->szObjName);
	ListBox_AddString(hwndList, buf);
	wsprintf(buf, "Roles = %s", poi->szRole);
	ListBox_AddString(hwndList, buf);
        wsprintf(buf, "State = %s", poi->szState);
	ListBox_AddString(hwndList, buf);
        wsprintf(buf, "Value = %s", poi->szValue);
	ListBox_AddString(hwndList, buf);
        wsprintf(buf, "Shortcut = %s", poi->szShortcut);
	ListBox_AddString(hwndList, buf);
        wsprintf(buf, "DefAction = %s", poi->szDefAction);
	ListBox_AddString(hwndList, buf);
	wsprintf(buf, "Help = %s", poi->szHelp);
	ListBox_AddString(hwndList, buf);

	wsprintf(buf, "Children = %s", poi->szChildren);
	ListBox_AddString(hwndList, buf);
        wsprintf(buf, "Parent = %s", poi->szParent);
	ListBox_AddString(hwndList, buf);
        wsprintf(buf, "Selection = %s", poi->szSel);
	ListBox_AddString(hwndList, buf);
        wsprintf(buf, "KeyboardFocus = %s", poi->szFocus);
	ListBox_AddString(hwndList, buf);

	
}

void UpdateObjectSelDlg(int iObjectSelTypeIndex, OACOBJINFO *poi)
{
        // Fill listbox with ifno in oi.
	ComboBox_SetCurSel(GetDlgItem(ghdlgObjectSel, DID_OBJECTSELTYPE), iObjectSelTypeIndex);
	grgoiObjectSel[iObjectSelTypeIndex] = *poi;
	FillOACList(GetDlgItem(ghdlgObjectSel, DID_OI), poi);

        // Move mouse pointer to new focal area.
        if (gfCursorTrack && poi->hr == S_OK  &&
            (iObjectSelTypeIndex == INDEX_SELECTION ||
             iObjectSelTypeIndex == INDEX_KEYBOARDFOCUS ||
             iObjectSelTypeIndex == INDEX_CARET))
            SetCursorPos(poi->xLeft, poi->yTop);

}

void UpdateSelectWindowDlg(OACOBJINFO *poi)
{
	FillOACList(GetDlgItem(ghdlgSelectWindow, DID_OI), poi);
}


/*****************************************************************************\
* FillListBox
*
* Fills the listbox in the Select Window dialog with the names of
* the child windows of the given window.
*
* Arguments:
*   hDlg     - Window handle of the dialog window
*   hwndList - Handle to the listbox within the dialog.
*   hwnd     - Parent whose children to enumerate.
*
* Returns:
*   VOID
\*****************************************************************************/

PRIVATE VOID
FillListBox(
    HWND hDlg,
    HWND hwndList,
    HWND hwnd
    )
{
    INT nIndex;

    //
    // First fill the list box with child windows
    //
    // Make sure we display the list box after things are added.
    //

    SendMessage(hwndList, WM_SETREDRAW, 0, 0);

    //
    // remember which dialog we are processing
    //

    ghwndDlgBeingFilled = hDlg;

    if (hwnd == NULL)
    {
	//
	// Enumerate the top level separately... gross unsymmetry, but
	// hey.
	//
	EnumWindows((WNDENUMPROC)AddOneWindow, (LPARAM)hwndList);
    }
    else
    {
	EnumChildWindows(hwnd, (WNDENUMPROC)AddOneWindow, (LPARAM)hwndList);
    }

    //
    // Now give the user a method of getting back to the parent. The space at
    // the beginning of the " [parent]" string identifies the entry as the
    // parent entry and makes it different from all the other entries since
    // the others start with a handle number of some sort.
    //

    nIndex = SendMessage(hwndList, LB_ADDSTRING, 0, (LPARAM)" [ parent... ]");
    SendMessage(hwndList, LB_SETITEMDATA, nIndex, (LONG)hwnd);

    //
    // Now do the redraw...
    //

    SendMessage(hwndList, WM_SETREDRAW, 1, 0);
}



/*****************************************************************************\
* AddOneWindow
*
* Gets the windows to add to the list of windows to spy on.
*
* Arguments:
*     HWND hwnd - handle of the window to add.
*     HWND hwndList - handle to the listbox.
*
* Returns:
*     TRUE - if a window was created.
\*****************************************************************************/
#define CCH_RGBUF   32

BOOL CALLBACK
AddOneWindow(
    HWND hwnd,
    HWND hwndList
    )
{
    CHAR rgBuf[CCH_RGBUF];
    INT nIndex;
    HWND htemp;

    //
    // Make sure we don't add any window that has anything to do with
    // the dialog or any other spy window
    //

    htemp = GetParent(hwnd);

    // Don't put console windows in the list since they can not be hooked
    if (GetClassName(hwnd, (LPTSTR) rgBuf, CCH_RGBUF) != 0 &&
	strcmp(rgBuf, szConsoleWindowClass) == 0 )
    {
	return 1;
    }

    // Don't put windows that belong to spy in the list
    if (hwnd == ghwndDlgBeingFilled || htemp == ghwndDlgBeingFilled
	|| hwnd == ghwndSpyApp
	|| htemp == ghwndSpyApp || hwnd == ghwndPrintf || htemp == ghwndPrintf
	|| hwnd == ghwndSpyHook || htemp == ghwndSpyHook)
    {
	return 1;
    }

    MakeWindowName(hwnd, rgBuf, CCH_RGBUF);

    nIndex = SendMessage(hwndList, LB_ADDSTRING, 0, (LPARAM)rgBuf);

    if (nIndex == LB_ERR || nIndex == LB_ERRSPACE)
	return 0;

    if (SendMessage(hwndList, LB_SETITEMDATA, nIndex, (LPARAM)hwnd) == LB_ERR)
	return 0;

    return 1;
}



/*****************************************************************************\
* MakeWindowName
*
* Builds the window name from the window handle passed in.
*
* Arguments:
*     HWND hwnd - handle to the window.
*     LPSTR lpString - String to put window name into.
*     INT nSTringLen - Length of window string.
*
* Returns:
*     VOID
\*****************************************************************************/

PRIVATE VOID
MakeWindowName(
    HWND hwnd,
    LPSTR lpString,
    INT nStringLen
    )
{
    wsprintf(lpString, "%8.8lX:", hwnd);

    if (hwnd == NULL || !IsWindow(hwnd))
    {
	lstrcat(lpString, "!!!");
    }
    else
    {
	GetWindowText(hwnd, lpString + 9, nStringLen - 9);
    }
}



/*****************************************************************************\
* FindHwndInListBox
*
* Gets the window from the list of windows in the listbox.
*
* Arguments:
*     HWND hwndList - handle to the listbox.
*     HWND hSpyWnd - handle to the spy window.
*
* Returns:
*     INT - Index to the window in the listbox. 
\*****************************************************************************/

PRIVATE INT
FindHwndInListBox(
    HWND hwndList,
    HWND hSpyWnd
    )
{
    CHAR rgBuf[9];
    INT nIndex;

    wsprintf(rgBuf, "%08lX", (LONG)hSpyWnd);

    nIndex = SendMessage(hwndList, LB_FINDSTRING, (WPARAM)-1, (LPARAM)rgBuf);

    if (nIndex == LB_ERR)
	nIndex = 0;

    return nIndex;
}



/*****************************************************************************\
* HighlightWindow
*
* Used to temporarily highlight the window that the user has selected from
* the Select Window dialog.  It does this by inverting a border around the
* window.
*
* Arguments:
*     HWND hwnd - handle to the selected window.
*     BOOL fdraw - whether to draw the window inverted on non-inverted.
*
* Returns:
*     VOID
\*****************************************************************************/

PRIVATE VOID
HighlightWindow(
    HWND hwnd,
    BOOL fDraw
    )
{
    HDC hdc;
    RECT rc;

    bBorderOn = fDraw;

    if (hwnd == NULL || !IsWindow(hwnd))
	return;

    hdc = GetWindowDC(hwnd);
    GetWindowRect(hwnd, &rc);
    OffsetRect(&rc, -rc.left, -rc.top);

    if (!IsRectEmpty(&rc))
    {
	PatBlt(hdc, rc.left, rc.top, rc.right - rc.left, DINV,  DSTINVERT);
	PatBlt(hdc, rc.left, rc.bottom - DINV, DINV,
	    -(rc.bottom - rc.top - 2 * DINV), DSTINVERT);
	PatBlt(hdc, rc.right - DINV, rc.top + DINV, DINV,
	    rc.bottom - rc.top - 2 * DINV, DSTINVERT);
	PatBlt(hdc, rc.right, rc.bottom - DINV, -(rc.right - rc.left),
	    DINV, DSTINVERT);
    }

    ReleaseDC(hwnd, hdc);
}



/*****************************************************************************\
* SelectWindowUpdateInfo
*
* Updates the informational fields in the Select Window dialog when
* a new window is selected from the hwnd listbox.
*
* Arguments:
*     HWND hDlg - handle to the select window dialog box.
*     HWND hwnd - handle to the new window selected.
*
* Returns:
*     VOID
\*****************************************************************************/

PRIVATE VOID
SelectWindowUpdateInfo(
    HWND hDlg,
    HWND hwnd
    )
{
    HWND hParent;
    DWORD dwStyle;
    RECT rc;
    CHAR szTemp[MAXSTRING];

    if (hwnd)
    {
	hParent = GetParent(hwnd);
	dwStyle = GetWindowLong(hwnd, GWL_STYLE);

	MakeWindowName(hwnd, szTemp, MAXSTRING);
	SetDlgItemText(hDlg, DID_SELWINWINDOW, szTemp);

	GetClassName(hwnd, szTemp, MAXSTRING);
	{
            HRESULT hr=S_OK;
            IUnknown FAR * punk=NULL;
            char szTemp1[128];

             //
             // If the window supported OAC, get and display full info in listbox.
             //

            hr = AccessibleObjectFromWindow(hwnd, OBJID_WINDOW, IID_IUnknown, (void **)&punk);
            if (punk && !IsHResultError(hr, "SelectWindowUpdateInfo, AccessibleObjectFromWindow"))
            {
                OACOBJINFO oi;
                wsprintf(szTemp1, "OAC, %s", szTemp);
                memset((void*)&oi, 0, sizeof(oi));
                hr = GetObjectInfo(punk, 0, &oi);
                IsHResultError(hr, "SelectWindowUpdateInfo, GetObjectInfo");
                UpdateSelectWindowDlg(&oi);
            }
            else
            {
		wsprintf(szTemp1, "Not OAC, %s", szTemp);
                ListBox_ResetContent(GetDlgItem(hDlg, DID_OI));
            }
            SetDlgItemText(hDlg, DID_SELWINCLASS, szTemp1);
            if (punk) punk->Release();
	}

	if (hParent)
	{
	    MakeWindowName(hParent, szTemp, MAXSTRING);
	    SetDlgItemText(hDlg, DID_SELWINPARENT, szTemp);
	}
	else
	{
	    SetDlgItemText(hDlg, DID_SELWINPARENT, "<No Parent>");
	}

	GetWindowRect(hwnd, &rc);
	wsprintf(szTemp, "(%d,%d)-(%d,%d) %dx%d", rc,
	    rc.right-rc.left, rc.bottom-rc.top);
	SetDlgItemText(hDlg, DID_SELWINRECT, szTemp);

	if (dwStyle & WS_POPUP)
	    wsprintf (szTemp, "%08lX: WS_POPUP", dwStyle);
	else if (dwStyle & WS_CHILD)
	    wsprintf (szTemp, "%08lX: WS_CHILD, ID: %lX", dwStyle,
	    GetWindowLong(hwnd, GWL_ID));
	else if (dwStyle & WS_ICONIC)
	    wsprintf (szTemp, "%08lX: WS_ICONIC", dwStyle);
	else
	    wsprintf (szTemp, "%08lX: WS_OVERLAPPED", dwStyle);

	SetDlgItemText(hDlg, DID_SELWINSTYLE, szTemp);
    }
    else
    {
	SetDlgItemText(hDlg, DID_SELWINWINDOW, "<Undefined>");
	SetDlgItemText(hDlg, DID_SELWINCLASS,  "<Undefined>");
	SetDlgItemText(hDlg, DID_SELWINPARENT, "<Undefined>");
	SetDlgItemText(hDlg, DID_SELWINRECT, "<Undefined>");
	SetDlgItemText(hDlg, DID_SELWINSTYLE, "<Undefined>");
    }
}



/*****************************************************************************\
* SelectWindowEnableFields
*
* Enables/disables the different fields in the Select Window dialog
* based on whether the user wants to spy on all windows or individually
* select one.
*
* Arguments:
*   HWND hwnd    - Dialog window handle.
*   BOOL fEnable - TRUE to enable the fields, FALSE to disable them.
*
* Returns:
*   VOID
\*****************************************************************************/

PRIVATE VOID
SelectWindowEnableFields(
    HWND hwnd,
    BOOL fEnable
    )
{
    EnableWindow(GetDlgItem(hwnd, DID_SELWINLIST), fEnable);
    EnableWindow(GetDlgItem(hwnd, DID_SELWINWINDOW), fEnable);
    EnableWindow(GetDlgItem(hwnd, DID_SELWINCLASS), fEnable);
    EnableWindow(GetDlgItem(hwnd, DID_SELWINPARENT), fEnable);
    EnableWindow(GetDlgItem(hwnd, DID_SELWINRECT), fEnable);
    EnableWindow(GetDlgItem(hwnd, DID_SELWINSTYLE), fEnable);
}



